package com.mushiny.redisson.utils;

import com.mushiny.redisson.execption.RedissonErrorCode;
import com.mushiny.redisson.execption.RedissonException;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import java.util.concurrent.TimeUnit;

/**
 * 描述信息
 *
 * @auther lxy0
 * @since 2023/8/7 13:27
 */
@Slf4j
@Component
public class RedissonLockUtil {


    public static final Long WAITTIME = 10000L;

    public static final Long RELEASETIME = -1L;
    @Autowired
    RedissonClient redissonClient;

    /**
     * @param key
     * @param lockType REENTRANT_LOCK:可重入锁   FAIR_LOCK:公平锁   READ_LOCK:红锁   WRITE_LOCK:读写锁
     * @return
     */
    public RLock getLockByKey(String key, int lockType) {
        if (ObjectUtils.isEmpty(key)) {
            key = Thread.currentThread().getName() + ":" + Thread.currentThread().getId();
        }
        if (ObjectUtils.isEmpty(lockType)) {
            lockType = 1;
        }
        switch (lockType) {
            case 1:
                return redissonClient.getLock(key);
            case 2:
                return redissonClient.getFairLock(key);
            case 3:
                return redissonClient.getReadWriteLock(key).readLock();
            case 4:
                return redissonClient.getReadWriteLock(key).writeLock();
            default:
                throw new RuntimeException("do not support lock type:" + lockType);
        }
    }

    /**
     * 手动加锁，锁的有效期默认30秒
     *
     * @param rLock
     */
    public void lock(RLock rLock) {
        rLock.lock();
    }

    /**
     * 手动解锁
     *
     * @param rLock
     */
    public void unlock(RLock rLock) {
        rLock.unlock();
    }


    /**
     * 当releaseTime传的是-1或者是空时，开启watch dog模式，会开启自动续锁
     * 非阻塞式
     *
     * @param waitTime    等待时间
     * @param releaseTime 锁有效时间
     * @param key
     * @param type
     * @return
     */
    public Tuple2<RLock, Boolean> tryLock(Long waitTime, Long releaseTime, String key, int type) {
        boolean lock = false;
        RLock rLock = null;
        if (ObjectUtils.isEmpty(waitTime)) {
            waitTime = WAITTIME;
        }
        if (ObjectUtils.isEmpty(releaseTime)) {
            releaseTime = RELEASETIME;
        }
        try {
            rLock = getLockByKey(key, type);
            lock = rLock.tryLock(waitTime, releaseTime, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return Tuples.of(rLock, lock);
    }


    /**
     * 手动释放 redissonLock 锁
     *
     * @param rLock
     * @return
     */
    public boolean releaseLock(RLock rLock) {
        String threadName = Thread.currentThread().getName() + ":" + Thread.currentThread().getId();
        try {
            rLock.unlock();
            if (!rLock.isLocked()) {
                log.debug("已删除锁{}成功，当前线程：{}", rLock.getName(), threadName);
                return true;
            } else if (rLock.isHeldByCurrentThread()) {
                log.debug("未删除锁{}，还需当前线程解锁{}次，当前线程：{}", rLock.getName(), rLock.getHoldCount(), threadName);
                return true;
            } else {
                log.warn("未删除锁{}，还需其他线程解锁{}次，当前线程：{}", rLock.getName(), rLock.getHoldCount(), threadName);
                return false;
            }
        } catch (Throwable e) {
            log.error("解锁{}异常,返回false，当前线程：{}", threadName, e);
            return false;
        }
    }
}
