package com.mushiny.pubTools.utils;

import com.mushiny.pubTools.execption.RedissonErrorCode;
import com.mushiny.pubTools.execption.RedissonException;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.util.concurrent.TimeUnit;

/**
 * 描述信息
 *
 * @auther lxy0
 * @since 2023/8/7 13:27
 */
@Slf4j
@Component
public class RedissonLockUtil {


    public static final Long WAITTIME = 12000L;

    public static final Long RELEASETIME = 14000L;
    @Autowired
    RedissonClient redissonClient;

    /**
     * @param key
     * @param lockType REENTRANT_LOCK:可重入锁   FAIR_LOCK:公平锁   READ_LOCK:红锁   WRITE_LOCK:读写锁
     * @return
     */
    public  RLock getLockByKey(String key, int lockType) {
        if (ObjectUtils.isEmpty(key)) {
            key = Thread.currentThread().getName() + ":" + Thread.currentThread().getId();
        }
        if (ObjectUtils.isEmpty(lockType)) {
            lockType = 1;
        }
        switch (lockType) {
            case 1:
                return redissonClient.getLock(key);
            case 2:
                return redissonClient.getFairLock(key);
            case 3:
                return redissonClient.getReadWriteLock(key).readLock();
            case 4:
                return redissonClient.getReadWriteLock(key).writeLock();
            default:
                throw new RuntimeException("do not support lock type:" + lockType);
        }
    }

    /**
     * 非阻塞式，在1s内尝试获取锁
     * 获取成功默认锁30s，过期看门狗自动续期
     *
     * @param key
     * @param waitTime
     * @param type
     * @return
     */
    public Boolean tryLock(Long waitTime, String key, int type) {
        boolean lock = false;
        if (ObjectUtils.isEmpty(waitTime)) {
            waitTime = WAITTIME;
        }
        try {
            RLock rLock = getLockByKey(key, type);
            lock = rLock.tryLock(waitTime, TimeUnit.MILLISECONDS);
            if (!lock) {
                log.error("分布式锁加锁:{}失败，请重新尝试");
                throw new RedissonException(RedissonErrorCode.METHOD_IN_EXECUTION);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return lock;
    }


    /**
     * 当releaseTime传的是-1或者是空时，开启watch dog模式，会开启自动续锁
     * 非阻塞式
     * @param waitTime    等待时间
     * @param releaseTime 锁有效时间
     * @param key
     * @param type
     * @return
     */
    public Boolean tryLock(Long waitTime, Long releaseTime, String key, int type) {
        boolean lock = false;
        if (ObjectUtils.isEmpty(waitTime)) {
            waitTime = WAITTIME;
        }
        if (ObjectUtils.isEmpty(releaseTime)) {
            releaseTime = RELEASETIME;
        }
        try {
            RLock rLock = getLockByKey(key, type);
            lock = rLock.tryLock(waitTime, releaseTime, TimeUnit.MILLISECONDS);
            if (!lock) {
                log.error("分布式锁加锁:{}失败，请重新尝试");
                throw new RedissonException(RedissonErrorCode.METHOD_IN_EXECUTION);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return lock;
    }


    /**
     * 手动释放 redissonLock 锁
     *
     * @param key
     * @return
     */
    public boolean releaseLock(String key, int type) {
        String threadName = Thread.currentThread().getName() + ":" + Thread.currentThread().getId();
        try {
            RLock rLock = getLockByKey(key, type);
            rLock.unlock();
            if (!rLock.isLocked()) {
                log.debug("已删除锁{}成功，当前线程：{}", key, threadName);
                return true;
            } else if (rLock.isHeldByCurrentThread()) {
                log.debug("未删除锁{}，还需当前线程解锁{}次，当前线程：{}", key, rLock.getHoldCount(), threadName);
                return true;
            } else {
                log.warn("未删除锁{}，还需其他线程解锁{}次，当前线程：{}", key, rLock.getHoldCount(), threadName);
                return false;
            }
        } catch (Throwable e) {
            log.error("解锁{}异常,返回false，当前线程：{}", key, threadName, e);
            return false;
        }
    }
}
