package com.mushiny.redisson.utils;

import cn.hutool.core.util.RandomUtil;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Random;


/**
 * 雪花算法的原理就是生成一个的 64 位比特位的 long 类型的唯一 id。开头一位固定0，41位时间戳，5位机器id，5位服务id，12位序号
 *
 * @auther lxy
 * @since 2023/8/9 13:57
 */
@Component
public class SnowFlakeUtil {
    /**
     * 起始时间戳，从2023-08-01开始生成
     */
    @Value("${snowflake.config.stamp:1690819200000L}")
    private final static long START_STAMP = 1690819200000L;

    /**
     * 序列号占用的位数 12
     */
    private final static long SEQUENCE_BIT = 12;

    /**
     * 机器标识占用的位数
     */
    private final static long MACHINE_BIT = 10;

    /**
     * 机器数量最大值
     */
    private final static long MAX_MACHINE_NUM = ~(-1L << MACHINE_BIT);

    /**
     * 序列号最大值
     */
    private final static long MAX_SEQUENCE = ~(-1L << SEQUENCE_BIT);

    /**
     * 每一部分向左的位移
     */
    private final static long MACHINE_LEFT = SEQUENCE_BIT;
    private final static long TIMESTAMP_LEFT = SEQUENCE_BIT + MACHINE_BIT;

    /**
     * 机器标识
     */
    @Value("${snowflake.config.machineId:160}")
    private long machineId;
    /**
     * 序列号
     */
    private long sequence = 0L;
    /**
     * 上一次时间戳
     */
    private long lastStamp = -1L;


    @Value("${snowflake.config.id.length:16")
    private int length = 16;

    @Value("${snowflake.config.id.strLength:16}")
    private int strLength = 25;


    public SnowFlakeUtil() {
        if (machineId > MAX_MACHINE_NUM || machineId < 0) {
            throw new RuntimeException("机器超过最大数量");
        }
    }

    /**
     * 产生下一个ID
     */
    public synchronized long nextId() {
        long currStamp = getNewStamp();
        if (currStamp < lastStamp) {
            throw new RuntimeException("时钟后移，拒绝生成ID！");
        }

        if (currStamp == lastStamp) {
            // 相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            // 同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currStamp = getNextMill();
            }
        } else {
            // 不同毫秒内，序列号置为0
            sequence = 0L;
        }
        lastStamp = currStamp;
        return transferLength((currStamp - START_STAMP) << TIMESTAMP_LEFT | machineId << MACHINE_LEFT | sequence);
    }


    /**
     * 产生下一个String类型的ID(随机5位字符串加上雪花算法id)
     */
    public synchronized String nextStrId() {
        long currStamp = getNewStamp();
        if (currStamp < lastStamp) {
            throw new RuntimeException("时钟后移，拒绝生成ID！");
        }
        if (currStamp == lastStamp) {
            // 相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            // 同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currStamp = getNextMill();
            }
        } else {
            // 不同毫秒内，序列号置为0
            sequence = 0L;
        }
        lastStamp = currStamp;
        return transferLength(String.valueOf((currStamp - START_STAMP) << TIMESTAMP_LEFT | machineId << MACHINE_LEFT | sequence));
    }

    private long getNextMill() {
        long mill = getNewStamp();
        while (mill <= lastStamp) {
            mill = getNewStamp();
        }
        return mill;
    }

    private long getNewStamp() {
        return System.currentTimeMillis();
    }


    /**
     * Long类型id转换长度
     *
     * @param id
     * @return
     */
    private long transferLength(Long id) {
        if (length < 20 && length > 15) {
            id = Long.parseLong(StringUtils.rightPad(id.toString(), length, "0"));
        }
        if (length > 19) {
            id = Long.parseLong(StringUtils.rightPad(id.toString(), 19, "0"));
        }
        return id;
    }

    /**
     * String类型id转换长度
     *
     * @param id
     * @return
     */
    private String transferLength(String id) {
        if (strLength > 16) {
            id = RandomUtil.randomString(strLength - id.length()) + id;
        }
        return id;
    }


    public static void main(String[] args) {
        SnowFlakeUtil snowFlakeUtil = new SnowFlakeUtil();
        Long nextId = snowFlakeUtil.nextId();
        String nextStrId = snowFlakeUtil.nextStrId();
        System.out.println("Long类型的为" + nextId + "长度为" + nextId.toString().length());
        System.out.println("String类型的为" + nextStrId + "长度为" + nextStrId.length());
    }


}
