package com.mushiny.redisson.aspect;

import com.mushiny.redisson.annotation.RedissonLock;
import com.mushiny.redisson.execption.RedissonErrorCode;
import com.mushiny.redisson.execption.RedissonException;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Method;


/**
 * 数据过滤，切面处理类
 *
 * @author lxy
 * @since 0.0.1
 */
@Aspect
@Component
@Slf4j
public class RedissonLockAspect {

    final String LOCKED = "REDISSON-LOCK";

    @Autowired
    RedissonClient redissonClient;


    @Pointcut("@annotation(com.mushiny.redisson.annotation.RedissonLock)")
    public void pointCut() {

    }

    /**
     * 公平锁切面
     */
    @Around("pointCut()")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        System.out.println("开始加锁");
        Signature sig = point.getSignature();
        MethodSignature methodSig = (MethodSignature) sig;
        Object target = point.getTarget();
        // 获取当前注解方法
        Method currentMethod = target.getClass().getMethod(methodSig.getName(), methodSig.getParameterTypes());
        RedissonLock lockAction = currentMethod.getAnnotation(RedissonLock.class);
        // 方法名+入参进行加锁
        StringBuilder args = new StringBuilder();
        //加锁key以REDISSONLOCK开头
        String key = LOCKED + currentMethod.getName() + "_" + args;
        //这里根据wms场景需要,放的是公平锁，也可自行修改为重入锁和红锁
        RLock lock = getLock(key, lockAction);
        // 具有Watch Dog 自动延期机制 默认续30s
        boolean locked = lock.tryLock(lockAction.waitTime(),-1,lockAction.unit());
        //加锁失败给前端返回码
        if (!locked) {
            log.error("加锁:{}失败，请重新尝试", currentMethod.getName());
            throw new RedissonException(RedissonErrorCode.METHOD_IN_EXECUTION);
        }
        try {
            // 加锁成功执行业务方法
            return point.proceed();
        } catch (Exception e) {
            log.error("执行方法:{}失败，请重新尝试", currentMethod.getName());
        } finally {
            //执行完成需要释放掉锁
            lock.unlock();
        }
        return null;
    }

    private RLock getLock(String key, RedissonLock lockAction) {
        switch (lockAction.lockType()) {
            case REENTRANT_LOCK:
                return redissonClient.getLock(key);

            case FAIR_LOCK:
                return redissonClient.getFairLock(key);

            case READ_LOCK:
                return redissonClient.getReadWriteLock(key).readLock();

            case WRITE_LOCK:
                return redissonClient.getReadWriteLock(key).writeLock();

            default:
                throw new RuntimeException("do not support lock type:" + lockAction.lockType().name());
        }
    }
}
