package com.mushiny.task.executor.core;

import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.function.Consumer;

/**
 * @author lihao
 */
public class TaskExecutor<T, U, R> {

    /**
     * 线程池
     */
    private ExecutorService pool;

    /**
     * 执行的任务数据
     */
    private TaskDefinition<T, U, R> task;

    private CountDownLatch countDownLatch;

    private final Long countDownTimeOut;

    private final TimeUnit timeUnit;

    public TaskExecutor(ThreadPoolExecutor pool, Long countDownTimeOut, TimeUnit timeUnit) {
        this.pool = pool;
        this.countDownTimeOut = countDownTimeOut;
        this.timeUnit = timeUnit;
    }


    /**
     * 初始化函数
     */
    public void init(TaskDefinition<T, U, R> task) {
        this.task = task;
    }

    public void init(BiFunction<T, U, R> handler, Collection<T> data, U context) {
        init(new TaskDefinition<>(data, handler, context));
    }

    public void init(BiFunction<T, U, R> handler, Collection<T> data, U context, Consumer<T> exceptionHandler) {
        init(new TaskDefinition<>(data, handler, context, exceptionHandler));
    }

    /**
     * 执行函数
     */
    public boolean execute() {
        TaskDefinition<T, U, R> t = this.task;
        Collection<T> data = t.getData();
        if (null == data || data.isEmpty()) {
            return false;
        }
        countDownLatch = new CountDownLatch(data.size());
        AtomicInteger failCount = new AtomicInteger();
        data.forEach(d -> pool.execute(() -> {
            try {
                Object apply = t.getHandler().apply(d, t.getContext());
            } catch (Exception ex) {
                if (Objects.nonNull(t.getExceptionHandler())) {
                    t.getExceptionHandler().accept(d);
                }
                failCount.addAndGet(1);
            } finally {
                // await超时会导致，loop里面继续执行，但是已经执行destroy方法将latch对象销毁
                if (Objects.nonNull(countDownLatch)) {
                    countDownLatch.countDown();
                }
            }
        }));
        try {
            return countDownLatch.await(countDownTimeOut, timeUnit)
                    && failCount.get() <= 0;
        } catch (InterruptedException e) {
            return false;
        } finally {
            this.destroy();
        }
    }

    /**
     * 销毁函数
     */
    public void destroy() {
        this.task = null;
        this.countDownLatch = null;
        this.pool = null;
    }





}
