/*
 * Copyright (c) 2020 牧星仓库管理系统 All rights reserved.
 *
 * http://www.mushiny.com
 *
 * 版权所有，侵权必究！
 */

package com.mushiny.wms.mfg.utils;

import com.mushiny.wms.mfg.constant.DateFormatPattern;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

/**
 * 时间工具类
 *
 * @author Gy
 * @since 4.0.0
 */
@Slf4j
public class DateTimeUtil {
    /**
     * 默认日期格式
     */
    private final static SimpleDateFormat defaultSimpleDateFormat = new SimpleDateFormat(DateFormatPattern.yyyy_MM_dd_HH_mm_ss);

    /**
     * 将日期字符串转化为日期类型
     * 格式：yyyy-MM-dd HH:mm:ss
     *
     * @param dateStr 日期字符串
     * @return
     */
    public static Date conventToDate(String dateStr) {
        try {
            if (StringUtils.isNotEmpty(dateStr)) {
                return defaultSimpleDateFormat.parse(dateStr + " 00:00:00");
            }
        } catch (ParseException e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("【时间转换失败】时间:" + dateStr);
        }
        return new Date();
    }

    /**
     * 将日期字符串转化为日期类型
     * 格式：yyyy-MM-dd HH:mm:ss
     *
     * @param dateStr 日期字符串
     * @return
     */
    public static Date conventToDate(String dateStr, String pattern) {
        try {
            if (StringUtils.isNotEmpty(dateStr)) {
                if (pattern.equals(DateFormatPattern.dd_MM_yyyy)) {
                    pattern = DateFormatPattern.yyyy_MM_dd;
                    String[] date = dateStr.split("-");
                    dateStr = date[2] + "-" + date[1] + "-" + date[0];
                }
                // 03-12-2022 12:00
                if (pattern.equals(DateFormatPattern.dd_MM_yyyy_HH_mm)) {
                    pattern = DateFormatPattern.yyyy_MM_dd_HH_mm;
                    String[] datetime = dateStr.split(" ");
                    String[] date = datetime[0].split("-");
                    dateStr = date[2] + "-" + date[1] + "-" + date[0] + " " + datetime[1];
                }
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
                Calendar calendar = new GregorianCalendar();
                calendar.setTime(simpleDateFormat.parse(dateStr));
                //calendar.add(Calendar.DATE, 1);
                return calendar.getTime();
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + dateStr);
        }
        return null;
    }

    /**
     * 格式化日期类型
     *
     * @param date    日期
     * @param pattern 日期格式
     * @return
     */
    public static String dateFormat(Date date, String pattern) {
        try {
            if (ObjectUtils.isNotEmpty(date)) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
                return simpleDateFormat.format(date);
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + e.getMessage());
        }
        return "";
    }

    /**
     * 格式化日期类型
     *
     * @param date     日期
     * @param pattern  日期格式
     * @param timeZone 时区
     * @return
     */
    public static String dateFormat(Date date, String pattern, String timeZone) {
        try {
            defaultSimpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            if (ObjectUtils.isNotEmpty(date)) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
                if (StringUtils.isNotBlank(timeZone)) {
                    simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timeZone));
                } else {
                    simpleDateFormat.setTimeZone(TimeZone.getDefault());
                }
                return simpleDateFormat.format(date);
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + e.getMessage());
        }
        return "";
    }

    /**
     * 格式化日期类型
     *
     * @param date    日期
     * @param pattern 日期格式
     * @return
     */
    public static String dateFormat(String date, String pattern) {
        try {
            if (StringUtils.isNotBlank(date)) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
                return simpleDateFormat.format(defaultSimpleDateFormat.parse(date));
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + e.getMessage());
        }
        return "";
    }

    /**
     * 格式化日期类型
     *
     * @param date     日期
     * @param pattern  日期格式
     * @param timeZone 时区
     * @return
     */
    public static String dateFormat(String date, String pattern, String timeZone) {
        try {
            defaultSimpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            if (StringUtils.isNotBlank(date)) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(pattern);
                if (StringUtils.isNotBlank(timeZone)) {
                    simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timeZone));
                } else {
                    simpleDateFormat.setTimeZone(TimeZone.getDefault());
                }
                return simpleDateFormat.format(defaultSimpleDateFormat.parse(date));
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + e.getMessage());
        }
        return "";
    }

    public static String dateFormat(LocalDateTime date, String pattern) {
        try {
            if (ObjectUtils.isNotEmpty(date)) {
                DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
                return formatter.format(date);
            }
        } catch (Exception e) {
            log.error("时间转化异常", e);
            throw new IllegalArgumentException("Time conversion failed,the time:" + e.getMessage());
        }
        return "";
    }

}
